<?php
/* --------------------------------------------------------------
   TrackingCodeFilter.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\TrackingCode\SqlCriteria;

use Gambio\Core\Criteria\SqlFilter;
use Webmozart\Assert\Assert;

/**
 * Class TrackingCodeFilter
 *
 * @package Gambio\Admin\TrackingCode\SqlCriteria
 */
class TrackingCodeFilter implements SqlFilter
{
    /**
     * @var string
     */
    private $attribute;
    
    /**
     * @var string
     */
    private $value;
    
    /**
     * @var string
     */
    private $operation;
    
    /**
     * @var string[]
     */
    private static $attributeColumnMapping = [
        'id'                    => 'orders_parcel_tracking_code_id',
        'orderId'               => 'order_id',
        'code'                  => 'tracking_code',
        'languageId'            => 'language_id',
        'parcelService.id'      => 'parcel_service_id',
        'parcelService.name'    => 'parcel_service_name',
        'parcelService.url'     => 'url',
        'parcelService.comment' => 'comment',
        'createdOn'             => 'creation_date',
    ];
    
    /**
     * @var string[]
     */
    private static $allowedForLikeOperation = [
        'code',
        'parcelService.name',
        'parcelService.url',
        'parcelService.comment',
    ];
    
    /**
     * @var string[]
     */
    private static $allowedForNumericOperation = [
        'id',
        'orderId',
        'languageId',
        'parcelService.id',
        'createdOn',
    ];
    
    
    /**
     * TrackingCodeFilter constructor.
     *
     * @param string $attribute
     * @param string $value
     * @param string $operation
     */
    private function __construct(string $attribute, string $value, string $operation)
    {
        $this->attribute = $attribute;
        $this->value     = $value;
        $this->operation = $operation;
    }
    
    
    /**
     * @param string $attribute
     * @param string $value
     * @param string $operation
     *
     * @return TrackingCodeFilter
     */
    public static function create(string $attribute, string $value, string $operation = 'like'): TrackingCodeFilter
    {
        $attributes = array_keys(self::$attributeColumnMapping);
        Assert::oneOf($attribute,
                      $attributes,
                      'Provided attribute needs to be one of: ' . implode(', ', $attributes) . '; Got: ' . $attribute);
        
        $operations = ['like', 'eq', 'gt', 'gte', 'lt', 'lte'];
        Assert::oneOf($operation,
                      $operations,
                      'Provided operation needs to be one of: ' . implode(', ', $operations) . '; Got: ' . $operation);
        
        if ($operation === 'like') {
            Assert::oneOf($attribute,
                          self::$allowedForLikeOperation,
                          'Provided attribute is not allowed for "like" operation.');
        } elseif (in_array($operation, ['gt', 'gte', 'lt', 'lte'])) {
            Assert::oneOf($attribute,
                          self::$allowedForNumericOperation,
                          'Provided attribute is not allowed for a numeric operation.');
        }
        
        return new self($attribute, $value, $operation);
    }
    
    
    /**
     * @return string
     */
    public function attribute(): string
    {
        return $this->attribute;
    }
    
    
    /**
     * @return string
     */
    public function value(): string
    {
        return $this->value;
    }
    
    
    /**
     * @return string
     */
    public function operation(): string
    {
        return $this->operation;
    }
    
    
    /**
     * @return string
     */
    public function column(): string
    {
        return self::$attributeColumnMapping[$this->attribute];
    }
}